<?php

namespace jl\plugins {

	use jl\exception\Exception;
	use jl\exception\InvalidArgumentException;
	use jl\Plugin;

	/**
	 * JLayer
	 *
	 * Upload files
	 *
	 * @package JohnCMS
	 * @subpackage JLayer
	 * @author Screamer
	 * @copyright 2013
	 */
	class Upload extends Plugin {

		/**
		 * (string) Name of input field in form
		 */
		const field_name = 'file';

		/**
		 * @var string Path to upload directory
		 */
		private $path;

		/**
		 * @var array Allowed extensions
		 */
		private $ext;

		/**
		 * @var array Allowed MIME-types
		 */
		private $mime;

		/**
		 * @var int Max file size in bytes
		 */
		private $size;

		/**
		 * Init
		 *
		 * @param string       $path Path to upload directory
		 * @param string|array $ext  Allowed extensions
		 * @param array        $mime Allowed MIME-types
		 * @param int          $size Max file size in Kilobytes
		 *
		 * @throws \jl\exception\Exception
		 * @throws \jl\exception\InvalidArgumentException
		 * @return void
		 */
		public function init($path = '', $ext = '', array $mime = array(), $size = 0) {
			if (!is_string($path)) {
				throw new InvalidArgumentException('path', 'string');
			}
			if (!is_array($ext) && !is_string($ext)) {
				throw new InvalidArgumentException('ext', 'string, array');
			}
			if (!is_array($mime)) {
				throw new InvalidArgumentException('mime', 'array');
			}
			if (!is_int($size)) {
				throw new InvalidArgumentException('size', 'int');
			}

			$path = !empty($path) ? rtrim($path, '\\/') . DIR_SEP : JL_MODULE_DIR . 'uploads' . DIR_SEP;
			if (!is_dir($path)) {
				throw new Exception('Wrong path to upload directory: ' . $path);
			}
			$this->path = $path;

			$this->mime = $this->jl->load_config('upload/mime', 'plugin');
			ksort($this->mime);
			if (!empty($ext)) {
				if (is_string($ext)) {
					$ext = preg_split('~[^a-z]~', $ext, -1, PREG_SPLIT_NO_EMPTY);
				}
			}
			$this->ext = !empty($ext) ? $ext : array_keys($this->mime);
			$this->mime = !empty($mime) ? $mime : $this->mime;

			$this->size = ($size !== 0 ? $size : intval($this->jl->functions->get_settings('flsz'))) * 1024;
		}

		/**
		 * Check file before upload
		 * Returns string with error if errors occured or array with the following items:
		 * [string:tmp_name] Path to file
		 * [string:name]     Name of file
		 * [string:ext]      Extension of file
		 *
		 * @param array $data Data of file
		 *
		 * @return string|array
		 */
		public function check(array $data) {
			$error = ''; // Occured errors
			$name  = ''; // File name
			$ext   = ''; // File extension
			if (isset($data['error'])) {
				switch ($data['error']) {
					case UPLOAD_ERR_CANT_WRITE:
						$error = $this->lng->upload_error_cant_write;
						break;
					case UPLOAD_ERR_INI_SIZE:
						$error = sprintf($this->lng->upload_error_ini_size, ini_get('upload_max_filesize'));
						break;
					case UPLOAD_ERR_NO_FILE:
						$error = $this->lng->upload_error_no_file;
						break;
					case UPLOAD_ERR_PARTIAL:
						$error = $this->lng->upload_error_partial;
						break;
					case UPLOAD_ERR_NO_TMP_DIR:
						$error = $this->lng->upload_error_not_tmp_dir;
						break;
					case UPLOAD_ERR_OK: // No errors occured
						if (isset($data['tmp_name']) && isset($data['name']) && isset($data['size']) && isset($data['type'])) {
							// Get extension
							$name = explode('.', $data['name']);
							$ext = array_pop($name);
							$name = implode('.', $name);
							// Check extension
							if (in_array($ext, $this->ext)) {
								// Check MIME-type
								$mime = isset($this->mime[$ext]) ? $this->mime[$ext] : null;
								if (!is_null($mime) && in_array($data['type'], $mime)) {
									// Check size
									if ($data['size'] <= $this->size) {
										// Filter Name
										$name = trim(str_replace(' ', '_', mb_strtolower($name)), '_');
										$name = $this->jl->functions->do_translit($name);
										$name = preg_replace('~([^a-z\_\d])~i', '', $name);
										$name = preg_replace('~([\_]{2,})~', '_', $name);
										if (empty($name)) {
											$error = $this->lng->wrong_file_name;
										}
									} else {
										$error = sprintf($this->lng->wrong_file_size, round($data['size'] / 1024), round($this->size / 1024));
									}
								} else {
									$error = sprintf($this->lng->wrong_file_ext, wordwrap(implode(', ', $this->ext), 75, '<br />'));
								}
							} else {
								$error = sprintf($this->lng->wrong_file_ext, wordwrap(implode(', ', $this->ext), 75, '<br />'));
							}
						}
						break;
				}
			}
			return !empty($error)
				? sprintf($this->lng->upload_error, $error)
				: array(
					'tmp_name' => $data['tmp_name'],
					'name'     => $name,
					'ext'      => $ext,
				);
		}

		/**
		 * Upload file
		 *
		 * @param string $location   Location of temp file
		 * @param string $name       Name of new file
		 * @param string $ext        Extension of new file
		 * @param string $upload_dir Path to upload dir
		 *
		 * @throws \jl\exception\Exception
		 * @throws \jl\exception\InvalidArgumentException
		 * @return boolean
		 */
		public function upload($location, $name, $ext, $upload_dir = '') {
			if (!is_string($location)) {
				throw new InvalidArgumentException('location', 'string');
			}
			if (!is_string($name)) {
				throw new InvalidArgumentException('name', 'ext');
			}
			if (!is_string($ext)) {
				throw new InvalidArgumentException('ext', 'string');
			}
			if (!is_string($upload_dir)) {
				throw new InvalidArgumentException('upload_dir', 'string');
			}

			if (empty($upload_dir)) {
				$upload_dir = $this->path;
			} else {
				$upload_dir = rtrim($upload_dir, '\\/') . DIR_SEP;
			}
			if (!is_dir($upload_dir)) {
				throw new Exception('Unable to upload file. Directory "' . $upload_dir . '" is not exists.');
			}
			if (!is_file($location)) {
				throw new Exception('Unable to upload file. Temp file "' . $location . '" is not exists');
			}
			$path = $upload_dir . $name . '.' . $ext;
			if (is_file($path)) {
				$path = $upload_dir . $name . '_' . time() . '.' . $ext;
			}

			return copy($location, $path);
		}

		/**
		 * Prepare upload form to display
		 *
		 * @param string        $label        Label value for field
		 * @param string        $field_name   Name of input field
		 * @param string        $error        Error message if exists
		 * @param string        $title        Title of page
		 * @param array|boolean $submit       Submit buttons.
		 *                                    Values:
		 *                                        false - no display;
		 *                                        true - display default;
		 *                                        array: (
		 *                                            ok     => [string:name, string:value],
		 *                                            cancel => [string:name, string:value],
		 *                                        )
		 *
		 * @throws \jl\exception\InvalidArgumentException
		 * @return string
		 */
		public function display_form($label = '', $field_name = '', $error = '', $title = '', $submit = false) {
			if (!is_string($label)) {
				throw new InvalidArgumentException('label', 'string');
			}
			if (!is_string($field_name)) {
				throw new InvalidArgumentException('field_name', 'string');
			}
			if (!is_string($error)) {
				throw new InvalidArgumentException('error', 'string');
			}
			if (!is_string($title)) {
				throw new InvalidArgumentException('title', 'string');
			}
			if (!is_array($submit) && !is_bool($submit)) {
				throw new InvalidArgumentException('submit', 'array, boolean');
			}

			if ($submit === true) {
				$submit = array(
					'ok' => array(
						'name' => 'ok',
						'value' => $this->lng->ok,
					),
					'cancel' => array(
						'name' => 'cancel',
						'value' => $this->lng->cancel,
					),
				);
			}
			return $this->tpl->load('display_form', array(
				'title'        => $title,
				'label'        => empty($label) ? $this->lng->input_label : $label,
				'error'        => $error,
				'field_name'   => empty($field_name) ? $field_name : self::field_name,
				'max_size'     => sprintf($this->lng->max_size, round($this->size / 1024)),
				'submit'       => is_array($submit) ? $submit : array(),
			));
		}

	}
}