<?php
namespace jl\plugins {

	use jl\exception\InvalidArgumentException;
	use jl\Plugin;
	use jl;

	/**
	 * JLayer
	 *
	 * Comments handler
	 *
	 * @package JohnCMS
	 * @subpackage JLayer
	 * @author Screamer
	 * @copyright 2013
	 */
	class Comments extends Plugin {

		/**
		 * @var string URI of controller
		 */
		private $uri;

		/**
		 * @var string Title of list of comments
		 */
		private $title;

		/**
		 * @var string Name of table with comments without prefix
		 */
		private $comments_table;

		/**
		 * @var string Name of table which contains commenting object without prefix
		 */
		private $object_table;

		/**
		 * @var string Name of field for count comments in object table
		 */
		private $counter_field;

		/**
		 * @var int Object identifier
		 */
		private $object_id;

		/**
		 * @var string Name of field which contains object identifier in objects table
		 */
		private $object_field;

		/**
		 * @var boolean Access to adm-actions
		 */
		private $allow_access;

		/**
		 * Init params
		 *
		 * @param string $uri            Uri of controller
		 * @param string $title          Title of list of comments
		 * @param string $comments_table Name of table with comments without prefix
		 * @param array  $access         Access list
		 * @param string $object_table   Name of table which contains commenting object without prefix
		 * @param int    $object_id      Object identifier
		 * @param string $object_field   Name of field which contains object identifier in objects table
		 * @param string $counter_field  Name of field for count comments in object table
		 *
		 * @throws \jl\exception\InvalidArgumentException
		 * @return Comments
		 */
		public function init(
			$uri,
			$title,
			$comments_table,
			array $access = array(),
			$object_table = '',
			$object_id = 0,
			$object_field = '',
			$counter_field = ''
		) {
			if (!is_string($uri)) {
				throw new InvalidArgumentException('uri', 'string');
			}
			if (!is_string($title)) {
				throw new InvalidArgumentException('title', 'string');
			}
			if (!is_string($comments_table)) {
				throw new InvalidArgumentException('comments_table', 'string');
			}
			if (!is_string($object_table)) {
				throw new InvalidArgumentException('object_table', 'string');
			}
			if (!is_int($object_id)) {
				throw new InvalidArgumentException('object_id', 'int');
			}
			if (!is_string($object_field)) {
				throw new InvalidArgumentException('object_field', 'string');
			}
			if (!is_string($counter_field)) {
				throw new InvalidArgumentException('counter_field', 'string');
			}

			$this->uri            = trim($uri, '\\/');
			$this->title          = $title;
			$this->comments_table = $this->jl->get_tables_prefix() . '_' . $comments_table;
			$this->object_table   = $this->jl->get_tables_prefix() . '_' . $object_table;
			$this->object_id      = $object_id;
			$this->object_field   = $object_field;
			$this->counter_field  = $counter_field;
			$this->allow_access   = in_array($this->jl->user->get_rights(), $access);
		}


		/**
		 * Display list of comments
		 *
		 * @param int    $arg      Argument
		 * @param string $arg_type Type of argument: empty - for display list; 'edit' - ID of comment for edit; 'remove' - ID of comment for remove;
		 *
		 *
		 * @return string
		 */
		public function display_list($arg = 0, $arg_type = '') {
			$arg = intval($arg);
			if ($arg_type == 'edit') {
				return $this->action_form($arg);
			} elseif ($arg_type == 'remove') {
				return $this->action_remove($arg);
			} else {
				$sql_postfix = $this->object_id == 0 ? "" : " WHERE `" . $this->comments_table . "`.`id` = '" . $this->object_id . "'";
				$total = $this->db->result("SELECT COUNT(*) FROM `" . $this->comments_table . "`" . $sql_postfix);
				$this->jl->plugin('pagination')->prepare(jl\anchor($this->uri), $total, $arg, $this->jl->user->get_data('kmess'));
				$form = $this->action_form();
				if ($total > 0) {
					$limit = $this->jl->plugin('pagination')->get_position();
					$result = $this->db->query(
						"SELECT `" . $this->comments_table . "`.`id` as `cid`, `" . $this->comments_table . "`.*, "
						. "`" . $this->jl->user->get_table() . "`.*"
						. "FROM `" . $this->comments_table . "` "
						. "LEFT JOIN `" . $this->jl->user->get_table() . "` "
						. "ON `" . $this->jl->user->get_table() . "`.`id` = `" . $this->comments_table . "`.`owner_id` "
						. $sql_postfix
						. "ORDER BY `time` DESC "
						. "LIMIT " . $limit['start'] . ", " . $limit['end']
					);
					$list = '';
					$i = 0;
					while ($item = $this->db->fetch_assoc($result)) {
						$item['owner_name'] = htmlspecialchars($item['owner_name'], ENT_QUOTES, 'UTF-8');
						$sub = array(
							'<a href="javascript:answer(\'message_field\', \'' . $item['owner_name'] . '\')">' . $this->lng->answer . '</a>'
						);
						if ($this->allow_access) {
							$sub[] = \jl\anchor($this->uri . '/' . $item['cid'] . '/edit', $this->lng->edit);
							$sub[] = \jl\anchor($this->uri . '/' . $item['cid'] . '/remove', $this->lng->remove);
						}
						$list .= $this->tpl->load('comment_item', array(
							'i'       => ($i % 2 ? 2 : 1),
							'comment' => $this->jl->functions->display_user(
								$item,
								false,
								true,
								false,
								false,
								'(' . $this->jl->functions->display_date(intval($item['time'])) . ')',
								$this->jl->functions->check_out($item['message'], 1, 1, 1),
								implode(' | ', $sub)
							),
						));
						$i++;
					}
					$this->db->free($result);
				} else {
					$list = $this->jl->functions->display_message($this->lng->list_empty);
				}
				$top_menu = array();
				if ($this->allow_access) {
					$top_menu[] = \jl\anchor($this->uri . '/0/remove', $this->lng->remove_all_comments);
				}
				return $this->tpl->load('comments', array(
					'title'      => $this->title,
					'top_menu'   => implode(' | ', $top_menu),
					'form'       => $form,
					'list'       => $list,
					'total'      => sprintf($this->lng->total, $total),
					'pagination' => $this->jl->plugin('pagination')->create_links(),
				));
			}
		}

		/**
		 * Remove comments
		 */
		private function action_remove($id = 0) {
			if (!$this->allow_access) {
				\jl\redirect();
			}
			$data = $id == 0 ? array() : $this->get_comment($id);
			if (!empty($_POST)) {
				if (isset($_POST['ok'])) {
					$this->remove_comment($id, (empty($data) ? 'sub_id' : 'id'), true);
				}
				\jl\redirect($this->uri);
			}
			return $this->jl->functions->confirm_message(
				(!empty($data) ? $this->lng->remove_comment : $this->lng->remove_all_comments),
				$this->lng->remove_confirm,
				$this->lng->remove,
				$this->lng->cancel
			);
		}

		/**
		 * Edit or add comment
		 */
		private function action_form($id = 0) {
			$id = intval($id);
			if ($id != 0) {
				if (!$this->allow_access) {
					\jl\redirect();
				}
				$data = $this->get_comment($id);
				if (empty($data)) {
					\jl\redirect();
				}
			}
			$error = array();
			$message = isset($data) ? $data['message'] : '';
			$name = $this->jl->user->get_id() != 0 ? $this->jl->user->get_data('name') : '';
			if (isset($_POST['send_message'])) {
				if ($id == 0) {
					if ($this->jl->user->get_id() != 0) {
						$anti_flood = $this->jl->functions->anti_flood();
						if ($anti_flood !== false) {
							$error['message'] = sprintf($this->lng->anti_flood, $anti_flood);
						}
					} else {
						$captcha_message = $this->lng->wrong_captcha;
						$captcha = isset($_POST['captcha']) ? $this->jl->functions->check_captcha($_POST['captcha']) : $captcha_message;
						if (is_string($captcha) || $captcha === false) {
							$error['captcha'] = $captcha_message;
						}
					}
				}
				if (empty($error)) {
					if ($id == 0) {
						if ($this->jl->user->get_id() == 0) {
							$name = isset($_POST['username']) ? trim($_POST['username']) : '';
						}
						if (mb_strlen($name) < 2 || strlen($name) > 50) {
							$error['name'] = sprintf($this->lng->wrong_string_len, 2, 25);
						}
					}
					$message = isset($_POST['message']) ? trim($_POST['message']) : $message;
					$message_len = mb_strlen($message);
					if ($message_len < 2 || $message_len > 1500) {
						$error['message'] = sprintf($this->lng->wrong_string_len, 2, 1500);
					}
					if (empty($error)) {
						if ($id == 0) {
							$this->db->query(
								"INSERT INTO `" . $this->comments_table . "` SET "
								. "`sub_id` = '" . $this->object_id . "', "
								. "`owner_id` = '" . $this->jl->user->get_id() . "', "
								. "`owner_name` = '" . $this->db->escape_string($name) . "', "
								. "`message` = '" . $this->db->escape_string($message) . "', "
								. "`time` = '" . time() . "'"
							);
							if ($this->object_id != 0) {
								// Update counter of comments
								$this->update_counter(1, $this->db->affected_rows());
							}
							$this->jl->functions->set_last_post_time($this->jl->user->get_id());
						} else {
							$this->db->query(
								"UPDATE `" . $this->comments_table . "` SET "
								. "`message` = '" . $this->db->escape_string($message) . "' "
								. "WHERE `id` = '" . $id . "'"
							);
						}
						\jl\redirect($this->uri);
					}
				}
			}
			return $this->tpl->load('message_form', array(
				'title'    => $id != 0 ? $this->lng->edit_comment : '',
				'back'     => $id != 0 ? \jl\anchor($this->uri, $this->lng->back) : '',
				'error'    => $error,
				'message'  => htmlspecialchars($message),
				'lng'      => $this->lng,
				'bb_panel' => $this->jl->functions->bb_panel('comments', 'message'),
				'username' => $this->jl->user->get_id() == 0 ? htmlspecialchars($name) : false,
				'captcha'  => $this->jl->user->get_id() != 0
					? ''
					: $this->jl->functions->get_captcha('captcha', (isset($error['captcha']) ? $error['captcha'] : '')),
			));
		}

		/**
		 * Get comment data from database
		 * @param int $id ID of comment
		 *
		 * @return array
		 */
		private function get_comment($id) {
			$result = $this->db->query("SELECT * FROM `" . $this->comments_table . "` WHERE `id` = '" . $id . "'");
			$data = $this->db->fetch_assoc($result);
			$this->db->free($result);
			return is_array($data) ? $data : array();
		}

		/**
		 * Remove comment(s)
		 *
		 * @param int    $id             Identifier
		 * @param string $by             Type of identifier: id|sub_id
		 * @param bool   $update_counter Update comments counter?
		 *
		 * @return void
		 */
		public function remove_comment($id, $by = 'id', $update_counter = true) {
			$id = intval($id);
			$by = $by == 'id' ? 'id' : 'sub_id';
			$this->db->query("DELETE FROM `" . $this->comments_table . "` WHERE `" . $by . "` = '" . $id . "'");
			if ($update_counter) {
				$this->update_counter(0, $this->db->affected_rows());
			}
		}

		/**
		 * Update counter of comments
		 *
		 * @param int $plus_minus Up/Down counter
		 * @param int $deal       Deal of comments
		 *
		 * @return void
		 */
		private function update_counter($plus_minus, $deal) {
			$deal = intval($deal);
			$this->db->query(
				"UPDATE `" . $this->object_table . "` SET "
				. "`" . $this->counter_field . "` = `" . $this->counter_field . "` " . ($plus_minus > 0 ? '+' : '-') . " " . $deal . " "
				. "WHERE `" . $this->object_field . "` = '" . $this->object_id . "'"
			);
		}

	}
}