<?php

namespace App\Rezka;

use DiDom\Document;
use Illuminate\Support\Facades\Http;

class FilmPageParser
{
    private string $protocol = 'https://';
    private string $baseUrl;
    private string $userAgent;

    public function __construct()
    {
        $this->baseUrl = 'rezkify.com';
        $this->userAgent = 'Mozilla/5.0 (Macintosh; Intel Mac OS X 10_15_7) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/98.0.4758.102 Safari/537.36';
    }

    public function getFilm(string $url = '')
    {
        $film = Http::withUserAgent($this->userAgent)
            ->get($this->protocol . $this->baseUrl . $url);
        /*        $cookies = [];
                $cookieDomain = '';
                foreach ($film->cookies()->getIterator() as $item) {
                    $cookies[$item->getName()] = $item->getValue();
                    $cookieDomain = $item->getDomain();
                }*/

        $pageHtml = $film->body();
        $document = new Document($pageHtml);
        $streams = $this->getActiveStreams($pageHtml);

        return [
            'id'           => $document->first('#post_id')?->getAttribute('value'),
            'name'         => $this->getName($document),
            'altName'      => $this->getAltName($document),
            'previewImage' => $this->getPreview($document),
            'detailImage'  => $this->getDetailImage($document),
            'description'  => $this->getDescription($document),
            'translations' => $this->getTranslations($document),
            'seasons'      => $this->getSeasons($document),
            'episodes'     => $this->getEpisodes($document),
            'streams'      => $streams ?? $this->getActiveSeriesStreams($pageHtml),
        ];
    }

    /**
     * Название
     */
    public function getName(Document $document): ?string
    {
        return $document->first('[itemprop=name]')?->text();
    }

    /**
     * Альтернативное название (обычно английское)
     */
    public function getAltName(Document $document): ?string
    {
        return $document->first('[itemprop=alternativeHeadline]')?->text();
    }

    /**
     * Получение переводов
     */
    public function getTranslations(Document $document): array
    {
        $translations = [];
        $translationList = $document->first('#translators-list');
        if ($translationList) {
            $items = $translationList->find('li');
            foreach ($items as $item) {
                $classes = $item->getAttribute('class') ?? '';
                $translations[] = [
                    'id'     => $item->getAttribute('data-translator_id'),
                    'name'   => $item->text(),
                    'active' => str_contains($classes, 'active'),
                ];
            }
        }

        // Для сериалов с одним переводом достаем название и id из других частей страницы
        if (empty($translations)) {
            $postInfo = $document->first('.b-post__info');
            $rows = $postInfo->find('tr');
            foreach ($rows as $row) {
                $tds = $row->find('td');
                if ($tds[0]?->text() === 'В переводе:') {
                    $html = $document->html();
                    $startPos = mb_stripos($html, 'initCDNSeriesEvents');
                    if ($startPos !== false) {
                        $substr = mb_substr($html, $startPos);
                        $substr = mb_substr($substr, 0, (mb_stripos($substr, ' {') + 2));
                        $id = trim(explode(',', $substr)[1]);
                    }
                    $translations[] = [
                        'id'     => $id ?? 0,
                        'name'   => $tds[1]?->first('td')?->text(),
                        'active' => true,
                    ];
                }
            }
        }

        return $translations;
    }

    /**
     * Получение сезонов
     */
    private function getSeasons(Document $document): array
    {
        $seasons = [];
        $seasonList = $document->find('.b-simple_season__item');
        foreach ($seasonList as $item) {
            $classes = $item->getAttribute('class') ?? '';
            $seasons[] = [
                'id'     => $item->getAttribute('data-tab_id'),
                'name'   => $item->text(),
                'active' => str_contains($classes, 'active'),
            ];
        }
        return $seasons;
    }

    /**
     * Получение серий для сериалов
     */
    private function getEpisodes(Document $document): array
    {
        $episodes = [];
        $episodeList = $document->find('.b-simple_episode__item');
        foreach ($episodeList as $item) {
            $classes = $item->getAttribute('class') ?? '';
            $episodes[$item->getAttribute('data-season_id')][] = [
                'id'         => $item->getAttribute('data-id'),
                'episode_id' => $item->getAttribute('data-episode_id'),
                'season_id'  => $item->getAttribute('data-season_id'),
                'name'       => $item->text(),
                'active'     => str_contains($classes, 'active'),
            ];
        }
        return $episodes;
    }

    /**
     * Сжатая обложка в плохом качестве
     */
    private function getPreview(Document $document): ?string
    {
        return $document->first('[itemprop=image]')?->getAttribute('src');
    }

    /**
     * Обложка в оригинальном качестве
     */
    private function getDetailImage(Document $document): ?string
    {
        return $document->first('[itemprop=image]')?->parent()?->getAttribute('href');
    }

    private function getDescription(Document $document): ?string
    {
        return $document->first('.b-post__description_text')?->text();
    }

    /**
     * Получение потоков для определенной серии
     * @param $id - id фильма
     * @param $translatorId - id перевода
     * @param $season - номер сезона
     * @param $episode - id серии
     * @return null|array
     */
    public function getStreamsByEpisode($id, $translatorId, $season, $episode): ?array
    {
        $film = Http::withUserAgent($this->userAgent)
            ->asForm()
            ->post($this->protocol . $this->baseUrl . '/ajax/get_cdn_series/', [
                // Сериал
                'id'            => $id,
                'translator_id' => $translatorId,
                'season'        => $season,
                'episode'       => $episode,
                'action'        => 'get_stream',
            ])
            ->json();

        if ($film['success'] === true && $film['url']) {
            return app(FilmParser::class)->getStreams($film['url']);
        }

        return [];
    }

    /**
     * Получение потоков с текущей активной страницы
     */
    private function getActiveStreams(string $document): ?array
    {
        $startPos = mb_stripos($document, 'initCDNMoviesEvents');
        if ($startPos !== false) {
            $substr = mb_substr($document, $startPos);
            $substr = mb_substr($substr, (mb_stripos($substr, ' {') + 2));
            $endPos = mb_stripos($substr, '});');
            $json = json_decode('{' . mb_substr($substr, 0, $endPos) . '}', true);
            if (!empty($json['streams'])) {
                return app(FilmParser::class)->getStreams($json['streams']);
            }
        }
        return null;
    }

    /**
     * Получение потоков с текущей активной страницы
     */
    private function getActiveSeriesStreams(string $document): ?array
    {
        $startPos = mb_stripos($document, 'initCDNSeriesEvents');
        if ($startPos !== false) {
            $substr = mb_substr($document, $startPos);
            $substr = mb_substr($substr, (mb_stripos($substr, ' {') + 2));
            $endPos = mb_stripos($substr, '});');
            $json = json_decode('{' . mb_substr($substr, 0, $endPos) . '}', true);
            if (!empty($json['streams'])) {
                return app(FilmParser::class)->getStreams($json['streams']);
            }
        }
        return null;
    }


    /**
     * Получение сезонов и серий для определенной озвучки сериала
     */
    public function getEpisodesFromSite(int $filmId, int $translationId): array
    {
        $film = Http::withUserAgent($this->userAgent)
            ->asForm()
            ->post($this->protocol . $this->baseUrl . '/ajax/get_cdn_series/', [
                'id'            => $filmId,
                'translator_id' => $translationId,
                'action'        => 'get_episodes',
            ])
            ->json();

        $episodesDocument = new Document($film['episodes']);
        $seasonsDocument = new Document($film['seasons']);

        return [
            'seasons'  => $this->getSeasons($seasonsDocument),
            'episodes' => $this->getEpisodes($episodesDocument),
            'streams'  => app(FilmParser::class)->getStreams($film['url']),
        ];
    }

    /**
     * Получаем видео для определенного перевода фильма
     */
    public function getStreamForFilmTranslation(int $filmId, int $translationId): array
    {
        $film = Http::withUserAgent($this->userAgent)
            ->asForm()
            ->post($this->protocol . $this->baseUrl . '/ajax/get_cdn_series/', [
                'id'            => $filmId,
                'translator_id' => $translationId,
                'action'        => 'get_movie',
            ])
            ->json();

        return [
            'streams' => app(FilmParser::class)->getStreams($film['url']),
        ];
    }
}
