<?php
declare(strict_types=1);

namespace Qa\Actions;

use Qa\Context;

class File extends AbstractAction
{
    public function __construct(Context $context)
    {
        parent::__construct($context);
        $this->title = 'Просмотр вложения';
    }

    public function invoke() : void
    {
        if (!preg_match('/(q|a)(\d+)/', $this->context->routeParam(), $m)) {
            $this->resultNotFound();
        }

        $type = $m[1];
        $id = intval($m[2]);
        if ($id <= 0) {
            $this->resultNotFound();
        }

        $table = ($type == 'q' ? 'qa_questions' : 'qa_answers');
        $attachmentName = $this->context->db()->table($table)
                ->where('id', $id)
                ->value('attachment_name');
        if (!$attachmentName) {
            $this->resultNotFound();
        }

        $filePath = QA_UPLOAD_PATH . "{$type}_$id.dat";
        if (!file_exists($filePath)) {
            $this->resultNotFound();
        }

        $this->forceDownload($filePath, $attachmentName);
    }

    private function forceDownload(string $filename, string $outputName) : void {
        $ext = strtolower(pathinfo($outputName, PATHINFO_EXTENSION) ?? '');
        $inlineMimes = [
            // Image
            'bmp' => 'image/bmp',
            'gif' => 'image/gif',
            'jpg' => 'image/jpeg',
            'jpeg' => 'image/jpeg',
            'png' => 'image/png',
            'svg' => 'image/svg+xml',
            'webp' => 'image/webp',
            // Audio
            'mid' => 'audio/midi',
            'midi' => 'audio/midi',
            'mp3' => 'audio/mpeg',
            'wav' => 'audio/x-wav',
            // Video
            'mp4' => 'video/mp4',
            'webm' => 'video/webm',
            // Text documents
            'css' => 'text/css',
            'csv' => 'text/csv',
            'htm' => 'text/html',
            'html' => 'text/html',
            'js' => 'application/x-javascript',
            'json' => 'application/json',
            'log' => 'text/plain',
            'pdf' => 'application/pdf',
            'php' => 'text/plain',
            'txt' => 'text/plain',
            'xml' => 'application/xml',
            'yaml' => 'text/yaml',
            'yml' => 'text/yaml',
        ];
        if (isset($inlineMimes[$ext])) {
            $mime = $inlineMimes[$ext];
            $disposition = 'inline';
        } else {
            $mime = 'application/octet-stream';
            $disposition = 'attachment';
        }

        if (ob_get_level()) {
            ob_end_clean();
        }
        header('Content-Type: ' . $mime);
        header("Content-Disposition: $disposition; filename=\"$outputName\"");
        header('Content-Transfer-Encoding: binary');
        header('Content-Length: ' . filesize($filename));
        readfile($filename);
    }
}
