<?php
declare(strict_types=1);

namespace Qa\Actions;

use Qa\Context;

class Comment extends AbstractAction
{
    public function __construct(Context $context)
    {
        parent::__construct($context);
        $this->title = 'Новый комментарий';
    }

    public function invoke() : void
    {
        $id = intval($this->context->routeParam() ?: 0);
        if ($id <= 0) {
            $this->resultNotFound();
        }

        $commentsEntity = $this->context->commentsEntity();
        if (!$commentsEntity->canComment()) {
            $this->resultForbidden();
        }

        $type = (($_REQUEST['t'] ?? 'q') == 'q') ? 'q' : 'a';
        if (!$commentsEntity->isParentEntityExists($id, $type)) {
            $this->resultNotFound();
        }
        $this->backUrl = $this->getBacklink($id, $type);
        $comment = [
            'parent_id' => $id,
            'parent_type' => $type,
            'backlink' => $this->backUrl
        ];

        $flood = $this->context->tools()->antiflood();
        if ($flood) {
            $this->resultTooManyRequests($flood);
        }

        $errors = [];
        if (isset($_POST['submit'])) {
            $this->processPostData($comment);
            $errors = $this->validateComment($comment);
            if (empty($errors)) {
                $commentId = $commentsEntity->insert($comment);
                $this->updateLastPost();
                $_SESSION['alert'] = [
                    'type' => 'alert-success',
                    'content' => 'Комментарий добавлен. <a class="alert-link" href="#c' . $commentId . '">Перейти</a>'
                ];
                header('Location: ' . $this->backUrl);
                return;
            }
        }

        $this->render('qa::comment', [
            'comment' => $comment,
            'errors' => $errors
        ]);
    }

    protected function getBacklink(int $id, string $type) : string
    {
        return '/qa/' . ($type == 'q' ? 'show' : 'findanswer') . '/' . $id;
    }

    protected function processPostData(array &$comment) : void
    {
        $comment['text'] = trim($_POST['text'] ?? '');
    }

    protected function validateComment(array &$comment) : array
    {
        $errors = [];
        if (empty($comment['text']))
            $errors[] = 'Вы не ввели ответ';
        elseif (mb_strlen($comment['text']) > 500)
            $errors[] = 'Длина ответа не должна превышать 500 символов';

        return $errors;
    }
}
