<?php
declare(strict_types=1);

namespace Qa\Dao;

use Johncms\System\Container\Factory;
use PDO;
use Psr\Container\ContainerInterface;

class Answers extends AbstractEntity implements VoteableEntity
{
    public function __invoke(ContainerInterface $container) : self
    {
        $this->tableName = 'qa_answers';
        return $this;
    }

    public function getById(int $id) : array
    {
        $sql = "SELECT *, UNIX_TIMESTAMP(`qa_answers`.`created_at`) as `answering_time`
                FROM `qa_answers`
                WHERE id = ?";
        $answer = $this->prepareAndExec($sql, [$id])->fetch(PDO::FETCH_ASSOC);
        if (empty($answer['id'])) {
            return [];
        }
        if ($answer['attachment_name']) {
            $answer['attachment'] = $this->fillAttachmentArray(
                    $answer['attachment_name'], 'a_' . $id);
        }
        return $answer;
    }

    public function getForQuestion(array &$question,
            int $specificAnswerId, int $start, int $limit) : array
    {
        $answers = [];
        $filterSql = ($specificAnswerId > 0) ? " AND `qa_answers`.`id` = '$specificAnswerId'" : '';
        $sql = "SELECT `qa_answers`.*,
                   u1.`name` as `author_name`, u1.`rights` as `author_rights`, u1.`lastdate`,
                   u2.`name` as `editor_name`,
                   UNIX_TIMESTAMP(`qa_answers`.`created_at`) as `answering_time`,
                   UNIX_TIMESTAMP(`qa_answers`.`edited_at`) as `editing_time`,
                   COALESCE(SUM(`qa_answer_votes`.`value`), 0) as `rating`
                FROM `qa_answers`
                LEFT JOIN `users` u1 ON `qa_answers`.`author_id` = u1.`id`
                LEFT JOIN `users` u2 ON `qa_answers`.`edited_by` = u2.`id`
                LEFT JOIN `qa_answer_votes` ON `qa_answers`.`id` = `qa_answer_votes`.`answer_id`
                WHERE `qa_answers`.`question_id` = ? $filterSql
                GROUP BY `qa_answers`.`id`
                ORDER BY `rating` DESC, `qa_answers`.`created_at` DESC
                LIMIT $start, $limit";
        $stmt = $this->prepareAndExec($sql, [$question['id']]);

        $comments = Factory::getContainer()->get(Comments::class);
        $newestTime = $question['asking_time'];
        while ($answer = $stmt->fetch()) {
            $newestTime = max($newestTime, $answer['answering_time']);
            $answer['userOnline'] = time() <= ($answer['lastdate'] + 300);
            $answer['canEdit'] = $this->canEdit($answer);
            $answer['canDelete'] = $this->canDelete($answer);
            $answer['canVote'] = $this->canVote($answer);
            if ($answer['attachment_name']) {
                $answer['attachment'] = $this->fillAttachmentArray(
                        $answer['attachment_name'], 'a_' . $answer['id']);
            }
            [$answer['commentsCount'], $answer['comments']] = $comments->getFor(intval($answer['id']), 'a');
            $answers[] = $answer;
        }
        $stmt->closeCursor();

        if ($specificAnswerId > 0) {
            $answersCount = count($answers);
        } else {
            $sql = "SELECT COUNT(*) FROM `qa_answers` WHERE `question_id` = ?";
            $answersCount = $this->prepareAndExec($sql, [$question['id']])->fetchColumn();
        }

        return [intval($answersCount), $answers, $newestTime];
    }

    public function getAnswers(bool $unreadMode, int $start, int $limit) : array
    {
        $user = $this->user();

        $filterSql = '';
        if ($unreadMode) {
            $userId = $user->id;
            $time = time();
            $filterSql =
                    " LEFT JOIN `qa_answers_read` ON `qa_answers`.`id` = `qa_answers_read`.`answer_id`
                            AND `qa_answers_read`.`user_id` = $userId
                      WHERE `qa_answers_read`.`user_id` IS NULL
                        AND `qa_answers`.`created_at` > DATE_SUB(FROM_UNIXTIME($time), INTERVAL 7 DAY)";
        }

        $comments = Factory::getContainer()->get(Comments::class);
        $answers = [];
        $sql = "SELECT COUNT(*) FROM `qa_answers` $filterSql";
        $answersCount = $this->pdo()->query($sql)->fetchColumn();
        $sql = "SELECT
                    `qa_answers`.*, `qa_questions`.`title`,
                    `users`.`name` as `author_name`,
                    `users`.`rights` as `author_rights`,
                    UNIX_TIMESTAMP(`qa_answers`.`created_at`) as `answering_time`,
                    UNIX_TIMESTAMP(`qa_answers`.`edited_at`) as `editing_time`,
                    GREATEST(`qa_answers`.`created_at`, COALESCE(`qa_answers`.`edited_at`, 0)) as `last_modification`
                FROM `qa_answers`
                LEFT JOIN `qa_questions` ON `qa_answers`.`question_id` = `qa_questions`.`id`
                LEFT JOIN `users` ON `qa_answers`.`author_id` = `users`.`id`
                $filterSql
                GROUP BY `qa_answers`.`id`
                ORDER BY `last_modification` DESC
                LIMIT $start, $limit";
        $stmt = $this->pdo()->query($sql);
        while ($answer = $stmt->fetch()) {
            if ($answer['attachment_name']) {
                $answer['attachment'] = $this->fillAttachmentArray(
                        $answer['attachment_name'], 'a_' . $answer['id']);
            }
            [$answer['commentsCount'], $answer['comments']] = $comments->getFor(intval($answer['id']), 'a');
            $answers[] = $answer;
        }
        $stmt->closeCursor();
        return [$answersCount, $answers];
    }

    public function markAsRead(int $id) : void
    {
        $this->executeUnreadQuery("`qa_answers_read`.`user_id` IS NULL AND ans.`question_id` = '$id'");
    }

    public function resetUnread() : void
    {
        $this->executeUnreadQuery('`qa_answers_read`.`answer_id` IS NULL');
    }

    private function executeUnreadQuery($whereClause) : void
    {
        $userId = $this->user()->id;
        $time = time();
        $sql = "INSERT INTO `qa_answers_read`(`answer_id`,  `question_id`,    `user_id`, `read_at`, `status`)
                SELECT                         ans.`id`,   ans.`question_id`,  $userId,  FROM_UNIXTIME($time), 0
                FROM `qa_answers` ans
                LEFT JOIN `qa_answers_read` ON ans.`id` = `qa_answers_read`.`answer_id`
                      AND `qa_answers_read`.`user_id` = $userId
                WHERE $whereClause
                  AND ans.`created_at` > DATE_SUB(FROM_UNIXTIME($time), INTERVAL 7 DAY)";
        $this->pdo()->exec($sql);
    }

    public function insert(array &$answer) : int
    {
        $user = $this->user();
        $questionId = intval($answer['question_id']);
        $answerId = $this->table()->insertGetId([
            'question_id' => $questionId,
            'author_id' => $user->id,
            'text' => $answer['text'],
            'attachment_name' => $answer['attachment_name'] ?: null,
            'created_at' => $this->db()->raw('FROM_UNIXTIME(' . time() . ')'),
            'edited_at' => null,
            'edited_by' => 0
        ]);

        // Непрочитанные ответы
        $this->pdo()->exec(
                "DELETE FROM `qa_answers_read`
                 WHERE `question_id` = '$questionId' AND `user_id` = '0'");
        $this->pdo()->exec(
                "UPDATE `qa_answers_read` SET
                    `status` = '1'
                 WHERE `question_id` = '$questionId' AND `status` = '0'");

        $this->db()->table('qa_answers_read')->insert([
            'answer_id' => $answerId,
            'question_id' =>  $questionId,
            'user_id' => $user->id,
            'status' => 0,
            'read_at' => $this->db()->raw('FROM_UNIXTIME(' . time() . ')')
        ]);
        return $answerId;
    }

    public function update(array &$answer) : void
    {
        $this->table()->where('id', $answer['id'])->update([
            'text' => $answer['text'],
            'attachment_name' => $answer['attachment_name'] ?: null,
            'edited_at' => $this->db()->raw('FROM_UNIXTIME(' . time() . ')'),
            'edited_by' => $this->user()->id
        ]);
    }

    public function delete(array &$answer) : void
    {
        $id = intval($answer['id']);
        if ($answer['attachment_name']) {
            $answer['attachment'] = $this->fillAttachmentArray(
                    $answer['attachment_name'], 'a_' . $answer['id']);
            if (file_exists($answer['attachment']['path'])) {
                unlink($answer['attachment']['path']);
            }
        }
        $sql = "DELETE `qa_comments` FROM `qa_answers`
                LEFT JOIN `qa_comments` ON `qa_comments`.`parent_id` = `qa_answers`.`id`
                WHERE `qa_comments`.`parent_id` = '$id' AND `qa_comments`.`parent_type`='a'";
        $this->pdo()->exec($sql);
        $this->table()->delete($id);
    }

    public function canAnswer() : bool
    {
        $user = $this->user();
        return $user->isValid() && !isset($user->ban['1']);
    }

    public function canEdit(array &$answer) : bool
    {
        if (!$this->canAnswer()) {
            return false;
        }
        $user = $this->user();
        $isSameUser = ($user->id == $answer['author_id']);
        $isCurrentUserIsModerator = ($user->isValid() && $user->rights >= 6);
        // Пользователь может изменять свои ответы в течение 15 минут
        $hasTimeToEdit = $answer['answering_time'] > (time() - self::DEFAULT_EDITING_TIME);
        return $isCurrentUserIsModerator || ($isSameUser && $hasTimeToEdit);
    }

    public function canDelete(array &$answer) : bool
    {
        return $this->canEdit($answer);
    }

    public function canVote(array &$answer) : bool
    {
        $user = $this->user();
        if (!$user->isValid() || isset($user->ban['1'])) {
            return false;
        }
        if ($user->id == $answer['author_id']) {
            return false;
        }
        return true;
    }

    public function hasVote(int $id) : bool
    {
        return $this->db()->table('qa_answer_votes')->where([
            ['answer_id', '=', $id],
            ['user_id', '=', $this->user()->id]
        ])->exists();
    }

    public function vote(int $id, int $value) : void
    {
        $this->db()->table('qa_answer_votes')->insert([
            'answer_id' =>  $id,
            'user_id' => $this->user()->id,
            'value' => $value,
            'voted_at' => $this->db()->raw('FROM_UNIXTIME(' . time() . ')')
        ]);
    }
}