<?php
declare(strict_types=1);

namespace Qa\Actions;

use Qa\Context;

class Ask extends AbstractAction
{
    public function __construct(Context $context)
    {
        parent::__construct($context);
        $this->title = 'Новый вопрос';
    }

    public function invoke() : void
    {
        $flood = $this->context->tools()->antiflood();
        if ($flood) {
            $this->resultTooManyRequests($flood);
        }

        $questionsEntity = $this->context->questionsEntity();
        if (!$questionsEntity->canAsk()) {
            $this->resultForbidden();
        }

        $question = [];
        $errors = [];
        if (isset($_POST['submit'])) {
            $this->processPostData($question);
            $errors = $this->validateQuestion($question);
            if (empty($errors)) {
                $id = $questionsEntity->insert($question);
                if ($question['attachment_name'] && $this->hasFile()) {
                    $this->finishUploadFile($id);
                }
                $this->updateLastPost();
                $_SESSION['alert'] = [
                    'type' => 'alert-success',
                    'content' => 'Вопрос добавлен'
                ];
                header("Location: /qa/show/$id");
                return;
            }
        }

        $this->render('qa::ask', [
            'question' => $question,
            'errors' => $errors
        ]);
    }

    protected function processPostData(array &$question) : void
    {
        $question['title'] = trim(preg_replace('#[^\P{C}\n]+#u', '', $_POST['title'] ?? ''));
        $question['tagsArray'] = trim(preg_replace('/\s*,\s*/', ',', $_POST['tags'] ?? ''));
        $question['tagsArray'] = array_unique(array_filter(explode(',', $question['tagsArray'])));
        $question['tags'] = implode(',', $question['tagsArray']);
        $question['text'] = trim($_POST['text'] ?? '');
        $question['notify'] = isset($_POST['notify']) ? 1 : 0;
        if (empty($question['attachment_name'])) {
            $question['attachment_name'] = false;
        }
        if ($this->hasFile()) {
            $info = pathinfo( $_FILES['file']['name']);
            $filename = mb_substr($info['filename'] ?? 'file', 0, 80);
            $ext = mb_substr($info['extension'] ?? '', 0, 10);
            $question['attachment_name'] = "$filename.$ext";
        }
    }

    /**
     * Завершает выгрузку файла на сервер
     * @param int $id
     * @return void
     */
    protected function finishUploadFile(int $id) : void
    {
        move_uploaded_file($_FILES['file']['tmp_name'], QA_UPLOAD_PATH . "q_$id.dat");
    }

    protected function validateQuestion(array &$question) : array
    {
        $errors = [];
        if (empty($question['title']))
            $errors[] = 'Вы не ввели заголовок вопроса';
        elseif (mb_strlen($question['title']) > 150)
            $errors[] = 'Длина заголовка вопроса не должна превышать 150 символов';

        if (empty($question['tagsArray']))
            $errors[] = 'Вы не ввели теги';
        elseif (count($question['tagsArray']) > 9)
            $errors[] = 'Количество тегов превышает 9';
        elseif (mb_strlen($question['tags']) > 80)
            $errors[] = 'Общая длина строки тегов не должна превышать 80 символов';

        if (empty($question['text']))
            $errors[] = 'Вы не ввели содержимое вопроса';
        elseif (mb_strlen($question['text']) > 10000)
            $errors[] = 'Длина содержимого вопроса не должна превышать 10000 символов';

        $config = $this->context->config();
        $maxFileSizeInBytes = $config['flsz'] * 1024;
        if (!$errors && $this->hasFile() && ($_FILES['file']['size'] >= $maxFileSizeInBytes)) {
            $errors[] = 'Файл превышает допустимый размер в ' . (format_size($maxFileSizeInBytes));
        }

        return $errors;
    }

    protected function hasFile() : bool
    {
        if (empty($_FILES['file'])) {
            return false;
        }
        $size = ($_FILES['file']['size'] ?? 0);
        return $size > 0;
    }
}
