#!/usr/bin/env php
<?php

set_time_limit(0);
error_reporting(-1);
ini_set('display_errors', 1);

require __DIR__ . '/vendor/autoload.php';

use Ratchet\ConnectionInterface;
use Ratchet\Http\HttpServer;
use Ratchet\MessageComponentInterface;
use Ratchet\Server\IoServer;
use Ratchet\WebSocket\WsServer;
use React\EventLoop\Factory as EventLoop;
use React\Socket\Server as SocketServer;
use React\ZMQ\Context as ZMQContext;


$tmp = sys_get_temp_dir() . DIRECTORY_SEPARATOR . 'trillium-ws';
if (is_file($tmp)) {
    $pid = file_get_contents($tmp);
    if (file_exists('/proc/' . $pid)) {
        exit(sprintf('Found already running daemon instance [pid = %s], aborting...', $pid));
    } else {
        unlink($tmp);
    }
}
$handle = fopen($tmp, 'w');
fwrite($handle, getmypid());
fclose($handle);

function run($zmqSocket, $reactHost, $reactPort) {
    $loop      = EventLoop::create();
    $context   = new ZMQContext($loop);
    $sub      = $context->getSocket(\ZMQ::SOCKET_REP);
    $sub->bind($zmqSocket);
    $ws        = new MessageComponent($sub);
    $sub->on('message', [$ws, 'onZmqMessage']);
    $socketServer = new SocketServer($loop);
    $socketServer->listen($reactPort, $reactHost);
    new IoServer(new HttpServer(new WsServer($ws)), $socketServer);
    $loop->run();

}


class MessageComponent implements MessageComponentInterface
{

    private $mailSubscribers = [];
    private $sub;


    public function __construct($sub)
    {
        $this->sub = $sub;
    }

    /**
     * {@inheritdoc}
     */
    public function onOpen(ConnectionInterface $conn)
    {
        // TODO: Implement onOpen() method.
    }

    /**
     * @param ConnectionInterface|\StdClass $conn
     */
    public function onClose(ConnectionInterface $conn)
    {
        foreach ($this->mailSubscribers as $from) {
            if ($from->resourceId == $conn->resourceId) {
                echo 'Detached mail subscriber' . "\n";
                unset($this->mailSubscribers[$conn->resourceId]);
                break;
            }
        }

    }

    /**
     * {@inheritdoc}
     */
    public function onError(ConnectionInterface $conn, \Exception $e)
    {
        $conn->close();
    }

    /**
     * {@inheritdoc}
     */
    public function onMessage(ConnectionInterface $from, $msg)
    {
        $msg = json_decode($msg, true);
        if (!is_array($msg)) {
            echo 'Wrong message. Malformed JSON' . "\n";
            return;
        }
        if (!array_key_exists('action', $msg) || !array_key_exists('params', $msg)) {
            echo 'Wrong message format: ' . join(', ', array_keys($msg)) . "\n";
            return;
        }
        if (!is_array($msg['params'])) {
            echo 'Wrong message format' . "\n";
            return;
        }
        switch ($msg['action']) {
            case 'mail_subscribe':
                if (!array_key_exists('user_id', $msg['params'])) {
                    echo 'Unable to subscribe. user_id is not specified' . "\n";
                    return;
                }
                if ($msg['params']['user_id']) {
                    echo 'Attached mail subscriber' . "\n";
                    $this->mailSubscribers[$msg['params']['user_id']] = $from;
                }
                break;
            default:
                echo 'Unknown action' . "\n";
                break;
        }
    }

    public function onZmqMessage($msg)
    {
        echo 'Received zmq message';
        var_dump($msg);
        $msg = json_decode($msg, true);
        if (!is_array($msg)) {
            $this->sub->send('Wrong message. Malformed JSON' . "\n");
            return;
        }
        if (!array_key_exists('action', $msg) || !array_key_exists('params', $msg)) {
            $this->sub->send('Wrong message format');
            return;
        }
        if (!is_array($msg['params'])) {
            $this->sub->send('Wrong message format');
            return;
        }
        switch ($msg['action']) {
            case 'new_mail_message':
                if (!array_key_exists('user_id', $msg['params'])) {
                    $this->sub->send('Unable to send notification. user_id is not specified');
                    return;
                }
                if (array_key_exists($msg['params']['user_id'], $this->mailSubscribers)) {
                    $this->mailSubscribers[$msg['params']['user_id']]->send(json_encode([
                        'event' => 'new_mail_message'
                    ]));
                    $this->sub->send('Notification was sent');
                } else {
                    $this->sub->send('Subscriber does not exists');
                }
                break;
            default:
                break;
        }
    }

}

run('tcp://127.0.0.1:5555', '0.0.0.0', 8888);
