<?php

/**
 * JLayer
 * Pagination
 * @package JohnCMS
 * @subpackage JLayer
 * @author Screamer
 * @copyright 2013
 */

class JL_Pagination {

	/**
	 * @var Number of current page
	 */
	protected $current_page = 1;

	/**
	 * @var Deal of items on page
	 */
	protected $per_page = 5;

	/**
	 * @var Base URL
	 */
	protected $base_url = '';

	/**
	 * @var Deal of all items
	 */
	protected $total_rows = 0;

	/**
	 * @var First page
	 */
	protected $base_page = 1;

	/**
	 * @var Deal of pages
	 */
	protected $num_pages = 0;

	/**
	 * @var Position of interval for items
	 */
	protected $position = array();

	/**
	 * Prepare parameters
	 * @param (string) $base_url Base URL
	 * @param (int) $total_rows Total items
	 * @param (int) $current_page Current page
	 * @param (int) $per_page Deal of items on one page
	 * @return (array) Start and end values (array('start' => '', 'end' => '')))
	 */
	public function __construct($base_url, $total_rows, $current_page, $per_page = 0) {
		$this->total_rows = intval($total_rows);
		$per_page = intval($per_page);
		if ($per_page !== 0) {
			$this->per_page = $per_page;
		}
		// Detect base url
		$this->base_url = rtrim($base_url, '/') .'/';
		// Detect current page
		$this->current_page = intval($current_page);
		// Set current page to base page
        if ($this->current_page < 1) {
            $this->current_page = $this->base_page;
        }
        // Calculate the deal of pages
        $this->num_pages = intval(ceil($this->total_rows / $this->per_page));
		// If the page number beyond the result range the show last page
        if ($this->current_page > $this->num_pages) {
            $this->current_page = $this->num_pages;
        }
        $this->position =  array(
			'start' => abs(intval($this->current_page * $this->per_page - $this->per_page)),
			'end'   => intval($this->per_page),
		);
	}

	/**
	 * Create links
	 * @return (string)
	 */
	public function create_links()
	{
        if ($this->total_rows == 0 || $this->per_page == 0 || $this->num_pages == 1) {
            return '';
        }
        $output = array();
        // Previous
        if (($this->current_page - 1) > $this->base_page) {
        	$output[$this->current_page - 1] = '&#8592;';
        }
        // Next
    	if (($this->current_page + 1) < $this->num_pages) {
    		$output[$this->current_page + 1] = '&#8594;';
    	}
        // First page
        $output[$this->base_page] = $this->current_page - 1 == $this->base_page ? "&#8592;" : '&laquo;';
        // Last page
        $output[$this->num_pages] = $this->current_page + 1 == $this->num_pages ? "&#8594;" : '&raquo;';
        // Current page
        $output[$this->current_page] = FALSE;
        ksort($output);
        $data = array(
        	'pages'    => $output,
        	'base_url' => $this->base_url,
        );
        return JL_Template::get_instance()->load('_JL_Pagination', $data);
	}

	/**
	 * Get position of start end end values
	 * @return (array) [start][end]
	 */
	public function get_position() {
		return $this->position;
	}

}